/**********************************************************************************************************************
 Thanks for running these splitter tests for me.

 This script is a mostly hands off script.  It does everything needed for the tests.  All you need to do is...
 1.  Please make sure that SSMS is in the "Grid" output mode when running this script.
 2.  Run this script.  It builds all of the objects it needs and produces the result set need at the end.
     Note that this script runs in TempDB so as not to take any chances with your data, sprocs, etc.
 3.  When the run is Please copy the second result set (including the column names) into a spreadsheet and
     sent the spreadsheet to me at jbmoden@ameritech.net.
 4.  This script also deletes all of the objects it created except for the final result set which can
     be deleted after you've sent me the spreadsheet.  I left it there so if something went wrong,
     you wouldn't have to rerun the whole test again.

It would also be helpful if you provided a brief description of your hardware and the version of SQL Server that
your running on.  This script does NOT automatically capture any information about your machine.

 Thanks again for your help.
 --Jeff Moden - 10 Apr 2011
**********************************************************************************************************************/

--===== Do this all in a nice, safe place that everyone has.
    USE TempDB;
GO
--=====================================================================================================================
--      Create a unit based Tally table for Nadrek's functions
--=====================================================================================================================
--===== If the Tally table already exists here, drop it to make reruns easier.
     IF OBJECT_ID('tempdb.dbo.Tally','U') IS NOT NULL
        DROP TABLE dbo.Tally;

--===== Create and populate the Tally table on the fly.
     -- This ISNULL function makes the column NOT NULL
     -- so we can put a Primary Key on it
 SELECT TOP 11000
        IDENTITY(BIGINT,1,1) AS N
   INTO dbo.Tally
   FROM sys.all_columns ac1
  CROSS JOIN sys.all_columns ac2
;
--===== Add a CLUSTERED Primary Key to maximize performance
  ALTER TABLE dbo.Tally
    ADD CONSTRAINT PK_Tally_N 
        PRIMARY KEY CLUSTERED (N) WITH FILLFACTOR = 100
;
--===== Allow the general public to use it
  GRANT SELECT ON dbo.Tally TO PUBLIC
;
GO
--=====================================================================================================================
--      Conditionally drop and recreate the Split (CLR) splitter function.
--=====================================================================================================================
-- Drop the CLR function if it exists
IF  EXISTS 
    (
    SELECT 1
    FROM sys.objects 
    WHERE
        [object_id] = OBJECT_ID(N'dbo.Split') 
        AND type_desc = N'CLR_TABLE_VALUED_FUNCTION'
    )
    DROP FUNCTION dbo.Split
GO
-- Drop the assembly if it exists
IF  EXISTS 
    (
    SELECT 1 
    FROM sys.assemblies
    WHERE
        name = N'Split' 
    )
    DROP ASSEMBLY Split;
GO
-- Create the assembly
CREATE ASSEMBLY [Split] AUTHORIZATION [dbo]
FROM 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
WITH
    PERMISSION_SET = SAFE;
GO
-- Create the function
CREATE FUNCTION dbo.Split
(
    @Input      NVARCHAR(MAX), 
    @Delimiter  NCHAR(1)
)
RETURNS TABLE 
(
    ItemNumber  INTEGER NULL,
    Item        NVARCHAR(4000) NULL
)
AS EXTERNAL NAME Split.UserDefinedFunctions.Split;
GO
--=====================================================================================================================
--      Conditionally drop and recreate the DelimitedSplit8K_T0 splitter function.
--=====================================================================================================================
     IF OBJECT_ID('dbo.DelimitedSplit8K_T0') IS NOT NULL
        DROP FUNCTION dbo.DelimitedSplit8K_T0;
GO
CREATE FUNCTION dbo.DelimitedSplit8K_T0
--===== Define I/O parameters
        (@pString VARCHAR(8000), @pDelimiter CHAR(1))
RETURNS TABLE WITH SCHEMABINDING AS
 RETURN
--===== "Inline" CTE Driven "Tally Table" produces values from 0 up to 10,000...
     -- enough to cover VARCHAR(8000)
  WITH E1(N) AS (
                 SELECT 1 UNION ALL SELECT 1 UNION ALL SELECT 1 UNION ALL 
                 SELECT 1 UNION ALL SELECT 1 UNION ALL SELECT 1 UNION ALL 
                 SELECT 1 UNION ALL SELECT 1 UNION ALL SELECT 1 UNION ALL SELECT 1
                ),                          --10E+1 or 10 rows
       E2(N) AS (SELECT 1 FROM E1 a, E1 b), --10E+2 or 100 rows
       E4(N) AS (SELECT 1 FROM E2 a, E2 b), --10E+4 or 10,000 rows max
 cteTally(N) AS (--==== This provides the "base" CTE and limits the number of rows right up front
                     -- for both a performance gain and prevention of accidental "overruns"
                 SELECT TOP (ISNULL(DATALENGTH(@pString),0)) ROW_NUMBER() OVER (ORDER BY (SELECT NULL)) FROM E4
                ),
cteStart(N1) AS (--==== This returns N+1 (starting position of each "element" just once for each delimiter)
                 select 1 union all -- does away with 0 base CTE, and the OR condition in one go!
                 SELECT t.N+1 FROM cteTally t WHERE (SUBSTRING(@pString,t.N,1) = @pDelimiter)
                )
--===== Do the actual split. The ISNULL/NULLIF combo handles the length for the final element when no delimiter is found.
 SELECT ItemNumber = ROW_NUMBER() OVER(ORDER BY s.N1),
        Item       = SUBSTRING(@pString,s.N1,ISNULL(NULLIF(CHARINDEX(@pDelimiter,@pString,s.N1),0)-s.N1,8000))
   FROM cteStart s
;
GO
--=====================================================================================================================
--      Conditionally drop and recreate the DelimitedSplit8K_T1 splitter function.
--=====================================================================================================================
     IF OBJECT_ID('dbo.DelimitedSplit8K_T1') IS NOT NULL
        DROP FUNCTION dbo.DelimitedSplit8K_T1;
GO
CREATE FUNCTION dbo.DelimitedSplit8K_T1
--===== Define I/O parameters
        (@pString VARCHAR(8000), @pDelimiter CHAR(1))
RETURNS TABLE WITH SCHEMABINDING AS
 RETURN
--===== "Inline" CTE Driven "Tally Table" produces values from 0 up to 10,000...
     -- enough to cover VARCHAR(8000)
  WITH E1(N) AS (
                 SELECT 1 UNION ALL SELECT 1 UNION ALL SELECT 1 UNION ALL 
                 SELECT 1 UNION ALL SELECT 1 UNION ALL SELECT 1 UNION ALL 
                 SELECT 1 UNION ALL SELECT 1 UNION ALL SELECT 1 UNION ALL SELECT 1
                ),                          --10E+1 or 10 rows
       E2(N) AS (SELECT 1 FROM E1 a, E1 b), --10E+2 or 100 rows
       E4(N) AS (SELECT 1 FROM E2 a, E2 b), --10E+4 or 10,000 rows max
 cteTally(N) AS (--==== This provides the "base" CTE and limits the number of rows right up front
                     -- for both a performance gain and prevention of accidental "overruns"
                 SELECT TOP (ISNULL(DATALENGTH(@pString),0)) ROW_NUMBER() OVER (ORDER BY (SELECT NULL)) FROM E4
                ),
cteStart(N1) AS (--==== This returns N+1 (starting position of each "element" just once for each delimiter)
                 SELECT 1 UNION ALL -- does away with 0 base CTE, and the OR condition in one go!
                 SELECT t.N+1 FROM cteTally t WHERE SUBSTRING(@pString,t.N,1) = @pDelimiter
                ),
cteLen(N1,L1) AS(--==== Return start and length (for use in substring)
                 SELECT s.N1,
                        ISNULL(NULLIF(CHARINDEX(@pDelimiter,@pString,s.N1),0)-s.N1,8000)
                   FROM cteStart s
                )
--===== Do the actual split. The ISNULL/NULLIF combo handles the length for the final element when no delimiter is found.
 SELECT ItemNumber = ROW_NUMBER() OVER(ORDER BY l.N1),
        Item       = SUBSTRING(@pString, l.N1, l.L1)
   FROM cteLen l
;
GO
--=====================================================================================================================
--      Conditionally drop and recreate the NadrekUnion0 splitter function.
--=====================================================================================================================
     IF OBJECT_ID('dbo.NadrekUnion0') IS NOT NULL
        DROP FUNCTION dbo.NadrekUnion0;
GO
 CREATE FUNCTION [dbo].[NadrekUnion0](@pString [varchar](8000), @pDelimiter [char](1))
RETURNS TABLE 
 RETURN
WITH 
cteTally(N) AS (--==== This provides the "zero base" and limits the number of rows right up front
                     -- for both a performance gain and prevention of accidental "overruns"
                 SELECT 0 UNION ALL
                 SELECT TOP (DATALENGTH(ISNULL(@pString,1))) N FROM dbo.Tally WITH (nolock) -- the 1 based + Union All is almost identical in performance to a 0 based TOP (Datalength()+1)
                ),
cteStart(N1) AS (--==== This returns N+1 (starting position of each "element" just once for each delimiter)
                 SELECT t.N+1
                   FROM cteTally t
                  WHERE (SUBSTRING(@pString,t.N,1) = @pDelimiter OR t.N = 0) 
                )
--===== Do the actual split. The ISNULL/NULLIF combo handles the length for the final element when no delimiter is found.
 SELECT ItemNumber = ROW_NUMBER() OVER(ORDER BY s.N1), -- not including StringNumber actually increases duration and CPU, but uses slightly fewer writes
        Item       = SUBSTRING(@pString,s.N1,ISNULL(NULLIF(CHARINDEX(@pDelimiter,@pString,s.N1),0)-s.N1,8000))
   FROM cteStart s WITH (nolock)
;
GO
--=====================================================================================================================
--      Conditionally drop and recreate the Nadrek0Based splitter function.
--=====================================================================================================================
     IF OBJECT_ID('dbo.Nadrek0Based') IS NOT NULL
        DROP FUNCTION dbo.Nadrek0Based;
GO
 CREATE FUNCTION [dbo].[Nadrek0Based](@pString [varchar](8000), @pDelimiter [char](1))
RETURNS TABLE 
 RETURN
WITH 
cteTally(N) AS (--==== This limits the number of rows right up front
                     -- for both a performance gain and prevention of accidental "overruns"
                 SELECT TOP (DATALENGTH(ISNULL(@pString,1))+1) N FROM dbo.Tally WITH (nolock) -- the 1 based + Union All is almost identical in performance to a 0 based TOP (Datalength()+1)
                ),
cteStart(N1) AS (--==== This returns N+1 (starting position of each "element" just once for each delimiter)
                 SELECT t.N+1
                   FROM cteTally t
                  WHERE (SUBSTRING(@pString,t.N,1) = @pDelimiter OR t.N = 0) 
                )
--===== Do the actual split. The ISNULL/NULLIF combo handles the length for the final element when no delimiter is found.
 SELECT ItemNumber = ROW_NUMBER() OVER(ORDER BY s.N1), -- not including StringNumber actually increases duration and CPU, but uses slightly fewer writes
        Item       = SUBSTRING(@pString,s.N1,ISNULL(NULLIF(CHARINDEX(@pDelimiter,@pString,s.N1),0)-s.N1,8000))
   FROM cteStart s WITH (nolock)
;
GO

-----------------------------------------------------------------------------------------------------------------------
--=====================================================================================================================
--      Conditionally drop and recreate the NEW optimized "DelimitedSplit8K" splitter function.
--=====================================================================================================================
     IF OBJECT_ID('dbo.DelimitedSplit8K') IS NOT NULL
        DROP FUNCTION dbo.DelimitedSplit8K;
GO
CREATE FUNCTION [dbo].[DelimitedSplit8K]
--===== Define I/O parameters
        (@pString VARCHAR(8000), @pDelimiter CHAR(1))
RETURNS TABLE WITH SCHEMABINDING AS
 RETURN
--===== "Inline" CTE Driven "Tally Table produces values from 0 up to 10,000...
     -- enough to cover VARCHAR(8000)
  WITH E1(N) AS (
                 SELECT 1 UNION ALL SELECT 1 UNION ALL SELECT 1 UNION ALL 
                 SELECT 1 UNION ALL SELECT 1 UNION ALL SELECT 1 UNION ALL 
                 SELECT 1 UNION ALL SELECT 1 UNION ALL SELECT 1 UNION ALL SELECT 1
                ),                          --10E+1 or 10 rows
       E2(N) AS (SELECT 1 FROM E1 a, E1 b), --10E+2 or 100 rows
       E4(N) AS (SELECT 1 FROM E2 a, E2 b), --10E+4 or 10,000 rows max
 cteTally(N) AS (--==== This provides the "zero base" and limits the number of rows right up front
                     -- for both a performance gain and prevention of accidental "overruns"
                 SELECT 0 UNION ALL
                 SELECT TOP (DATALENGTH(ISNULL(@pString,1))) ROW_NUMBER() OVER (ORDER BY (SELECT NULL)) FROM E4
                ),
cteStart(N1) AS (--==== This returns N+1 (starting position of each "element" just once for each delimiter)
                 SELECT t.N+1
                   FROM cteTally t
                  WHERE (SUBSTRING(@pString,t.N,1) = @pDelimiter OR t.N = 0) 
                )
--===== Do the actual split. The ISNULL/NULLIF combo handles the length for the final element when no delimiter is found.
 SELECT ItemNumber = ROW_NUMBER() OVER(ORDER BY s.N1),
        Item       = SUBSTRING(@pString,s.N1,ISNULL(NULLIF(CHARINDEX(@pDelimiter,@pString,s.N1),0)-s.N1,8000))
   FROM cteStart s
;
GO
--=====================================================================================================================
--      Conditionally drop and recreate a View that will allow us to use NEWID() within a function so we can make
--      Random numbers in a function and create a function that will create constrained randomized CSV element rows.
--=====================================================================================================================
--===== Conditionally drop the objects in the code below to make reruns easier
     IF OBJECT_ID('TempDB.dbo.iFunction'  ,'V' ) IS NOT NULL DROP VIEW     dbo.iFunction;
     IF OBJECT_ID('TempDB.dbo.CreateCsv8K','IF') IS NOT NULL DROP FUNCTION dbo.CreateCsv8K;
GO
 CREATE VIEW dbo.iFunction AS 
/**********************************************************************************************************************
 Purpose:
 This view is callable from UDF's which allows us to indirectly get a NEWID() within a function where we can't do such
 a thing directly in the function.  This view also solves the same problem for GETDATE().

 Usage:
 SELECT MyNewID FROM dbo.iFunction; --Returns a GUID
 SELECT MyDate  FROM dbo.iFunction; --Returns a Date


 Revision History:
 Rev 00 - 06 Jun 2004 - Jeff Moden - Initial creation
 Rev 01 - 06 Mar 2011 - Jeff Moden - Formalize code.  No logic changes. 
**********************************************************************************************************************/
SELECT MyNewID = NEWID(),
       MyDate  = GETDATE();
GO 
 CREATE FUNCTION dbo.CreateCsv8K
/**********************************************************************************************************************
 Purpose:
 Create a CSV table result with a programable number of rows, elements per row, minimum # of characters per element,
 and maximum characters per element.  The element size is random in nature constrained by the min and max characters
 per element.

 Usage:
 SELECT * FROM dbo.CreateCsv8K(@pNumberOfRows, @pNumberOfElementsPerRow, @pMinElementwidth, @pMaxElementWidth)

 Dependencies:
 1. View: dbo.iFunction (Produces a NEWID() usable from within a UDF)

 Programmer's Notes:
 1. The randomness of the elements prevents the delimiters for showing up in the same position for each row so that
    SQL Server won't figure that out and cache the information making some splitting techniques seem faster than they
    really are.
 2. No validation or constraints have been place on the input parameters so use with caution.  This code can generate
    a lot of data in a couple of heart beats.

 Revision History:
 Rev 00 - 11 May 2007 - Jeff Moden - Initial creation - Only returned one row and wasn't programmable.
 Rev 01 - 26 Jul 2009 - Jeff Moden - Added programmable variables but would only go to 20 characters wide.
 Rev 02 - 06 Mar 2011 - Jeff Moden - Converted to iTVF, added minimum element width, and made it so elements can be
                                     virtually any size.
**********************************************************************************************************************/
--===== Declare the I/0
        (
        @pNumberOfRows           INT,
        @pNumberOfElementsPerRow INT,
        @pMinElementwidth        INT,
        @pMaxElementWidth        INT
        )
RETURNS TABLE 
     AS
 RETURN
--===== This creates and populates a test table on the fly containing a
     -- sequential column and a randomly generated CSV Parameter column.
 SELECT TOP (@pNumberOfRows) --Controls the number of rows in the test table
        ISNULL(ROW_NUMBER() OVER (ORDER BY(SELECT NULL)),0) AS RowNum,
        CSV =
        (--==== This creates each CSV
         SELECT CAST(
                    STUFF( --=== STUFF get's rid of the leading comma
                         ( --=== This builds CSV row with a leading comma
                          SELECT TOP (@pNumberOfElementsPerRow) --Controls the number of CSV elements in each row
                                 ','
                               + LEFT(--==== Builds random length variable within element width constraints
                                      LEFT(REPLICATE('1234567890',CEILING(@pMaxElementWidth/10.0)), @pMaxElementWidth),
                                      ABS(CHECKSUM((SELECT MyNewID FROM dbo.iFunction)))
                                            % (@pMaxElementWidth - @pMinElementwidth + 1) + @pMinElementwidth
                                     )
                            FROM sys.All_Columns ac3            --Classic cross join pseudo-cursor
                           CROSS JOIN sys.All_Columns ac4       --can produce row sets up 16 million.
                           WHERE ac3.Object_ID <> ac1.Object_ID --Without this line, all rows would be the same.
                             FOR XML PATH('')
                         )
                    ,1,1,'')
                AS VARCHAR(8000))
        )
   FROM sys.All_Columns ac1      --Classic cross join pseudo-cursor
  CROSS JOIN sys.All_Columns ac2 --can produce row sets up 16 million rows
;
GO
--=====================================================================================================================
--      Conditionally drop and recreate the TestResults table
--=====================================================================================================================
--===== Conditionally drop and create the TestResults table
     IF OBJECT_ID('dbo.TestResults','U') IS NOT NULL DROP TABLE dbo.TestResults;

 CREATE TABLE dbo.TestResults
        (
        RowNum           INT IDENTITY(1,1) PRIMARY KEY CLUSTERED,
        SplitterName     VARCHAR(50),
        NumberOfRows     INT,
        NumberOfElements INT,
        MinElementLength    INT,
        MaxElementLength    INT,
        Duration         DECIMAL(9,5),
        MinLength        INT,
        AvgLength        INT,
        MaxLength        INT
        );
GO
--=====================================================================================================================
--      Conditionally drop and recreate the stored procedure that tests each function and records the results.
--=====================================================================================================================
--===== Conditionally drop and create the TestResults table
     IF OBJECT_ID('dbo.TestEachFunction','P') IS NOT NULL DROP PROCEDURE dbo.TestEachFunction;
GO
CREATE PROCEDURE dbo.TestEachFunction
/**********************************************************************************************************************
 Purpose:
 Given the number of rows andelements this testing is for, the stored procedure will test each of the split function
 for duration and record the results in an table called dbo.TestResults in the current DB (which should be TempDB).

 Revision History:
 Rev 00 - 10 Apr 2011 - Jeff Moden - Initial release for testing
**********************************************************************************************************************/
--===== Declare the I/O parameters
        @pNumberOfRows     INT,
        @pNumberOfElements INT,
        @pMinElementLength INT,
        @pMaxElementLength INT
     AS

--=====================================================================================================================
--      Presets
--=====================================================================================================================
--===== Suppress the auto-display of rowcounts for appearance and speed
    SET NOCOUNT ON;

--===== Declare some obviously named local variables
DECLARE @StartTime DATETIME,
        @EndTime   DATETIME,
        @Message   SYSNAME,
        @MinLength INT,
        @AvgLength INT,
        @MaxLength INT;

--===== Preset and display the current run message
 SELECT @Message = '========== ' 
                 + CAST(@pNumberOfRows     AS VARCHAR(10)) + ' Rows, '
                 + CAST(@pMinElementLength AS VARCHAR(10)) + ' MinElementSize, '
                 + CAST(@pMaxElementLength AS VARCHAR(10)) + ' MaxElementSize, '
                 + CAST(@pNumberOfElements AS VARCHAR(10)) + ' Elements '
                 + '==========';

RAISERROR(@Message,10,1) WITH NOWAIT;

--===== Calculate some statistics for the condition of the data
 SELECT @MinLength = MIN(DATALENGTH(CSV)), 
        @AvgLength = AVG(DATALENGTH(CSV)),
        @MaxLength = MAX(DATALENGTH(CSV))
   FROM dbo.Csv8K;
   
--===== Conditionally drop all test result tables =====================================================================
RAISERROR('Dropping temp tables...',10,1) WITH NOWAIT;
IF OBJECT_ID('tempdb..#Split')               IS NOT NULL DROP TABLE #Split;
IF OBJECT_ID('tempdb..#DelimitedSplit8K_T0') IS NOT NULL DROP TABLE #DelimitedSplit8K_T0;
IF OBJECT_ID('tempdb..#DelimitedSplit8K_T1') IS NOT NULL DROP TABLE #DelimitedSplit8K_T1;
IF OBJECT_ID('tempdb..#NadrekUnion0')        IS NOT NULL DROP TABLE #NadrekUnion0;
IF OBJECT_ID('tempdb..#Nadrek0Based')        IS NOT NULL DROP TABLE #Nadrek0Based;
IF OBJECT_ID('tempdb..#DelimitedSplit8K')    IS NOT NULL DROP TABLE #DelimitedSplit8K;

--=====================================================================================================================
--      Run the tests
--=====================================================================================================================
--===== Split (CLR) =================================================================================================
RAISERROR('Testing Split...',10,1) WITH NOWAIT;
DBCC FREEPROCCACHE;

--===== Start the timer
 SELECT @StartTime = GETDATE();

--===== Run the test
 SELECT csv.RowNum, split.ItemNumber, split.Item
   INTO #Split
   FROM dbo.CSV8K csv
  CROSS APPLY dbo.Split(csv.CSV,',') split;

--===== Stop the timer and record the test
 SELECT @EndTime = GETDATE();
 
 INSERT INTO dbo.TestResults
        (SplitterName, NumberOfRows, NumberOfElements, MinElementLength, MaxElementLength, Duration, MinLength, AvgLength, MaxLength)
 SELECT 'Split', 
        @pNumberOfRows, 
        @pNumberOfElements,
        @pMinElementLength,
        @pMaxElementLength,
        DATEDIFF(ms,@StartTime,@EndTime)/1000.0,
        MinLength = @MinLength,
        AvgLength = @AvgLength,
        MaxLength = @MaxLength
;

--===== DelimitedSplit8K_T0 =================================================================================================
RAISERROR('Testing DelimitedSplit8K_T0...',10,1) WITH NOWAIT;
DBCC FREEPROCCACHE;

--===== Start the timer
 SELECT @StartTime = GETDATE();

--===== Run the test
 SELECT csv.RowNum, split.ItemNumber, split.Item
   INTO #DelimitedSplit8K_T0
   FROM dbo.CSV8K csv
  CROSS APPLY dbo.DelimitedSplit8K_T0(csv.CSV,',') split;

--===== Stop the timer and record the test
 SELECT @EndTime = GETDATE();
 
 INSERT INTO dbo.TestResults
        (SplitterName, NumberOfRows, NumberOfElements, MinElementLength, MaxElementLength, Duration, MinLength, AvgLength, MaxLength)
 SELECT 'DelimitedSplit8K_T0', 
        @pNumberOfRows, 
        @pNumberOfElements,
        @pMinElementLength,
        @pMaxElementLength,
        DATEDIFF(ms,@StartTime,@EndTime)/1000.0,
        MinLength = @MinLength,
        AvgLength = @AvgLength,
        MaxLength = @MaxLength
;
--===== DelimitedSplit8K_T1 =================================================================================================
RAISERROR('Testing DelimitedSplit8K_T1...',10,1) WITH NOWAIT;
DBCC FREEPROCCACHE;

--===== Start the timer
 SELECT @StartTime = GETDATE();

--===== Run the test
 SELECT csv.RowNum, split.ItemNumber, split.Item
   INTO #DelimitedSplit8K_T1
   FROM dbo.CSV8K csv
  CROSS APPLY dbo.DelimitedSplit8K_T1(csv.CSV,',') split;

--===== Stop the timer and record the test
 SELECT @EndTime = GETDATE();
 
 INSERT INTO dbo.TestResults
        (SplitterName, NumberOfRows, NumberOfElements, MinElementLength, MaxElementLength, Duration, MinLength, AvgLength, MaxLength)
 SELECT 'DelimitedSplit8K_T1', 
        @pNumberOfRows, 
        @pNumberOfElements,
        @pMinElementLength,
        @pMaxElementLength,
        DATEDIFF(ms,@StartTime,@EndTime)/1000.0,
        MinLength = @MinLength,
        AvgLength = @AvgLength,
        MaxLength = @MaxLength
;
--===== NadrekUnion0 =================================================================================================
RAISERROR('Testing NadrekUnion0...',10,1) WITH NOWAIT;
DBCC FREEPROCCACHE;

--===== Start the timer
 SELECT @StartTime = GETDATE();

--===== Run the test
 SELECT csv.RowNum, split.ItemNumber, split.Item
   INTO #NadrekUnion0
   FROM dbo.CSV8K csv
  CROSS APPLY dbo.NadrekUnion0(csv.CSV,',') split;

--===== Stop the timer and record the test
 SELECT @EndTime = GETDATE();
 
 INSERT INTO dbo.TestResults
        (SplitterName, NumberOfRows, NumberOfElements, MinElementLength, MaxElementLength, Duration, MinLength, AvgLength, MaxLength)
 SELECT 'NadrekUnion0', 
        @pNumberOfRows, 
        @pNumberOfElements,
        @pMinElementLength,
        @pMaxElementLength,
        DATEDIFF(ms,@StartTime,@EndTime)/1000.0,
        MinLength = @MinLength,
        AvgLength = @AvgLength,
        MaxLength = @MaxLength
;

--===== Nadrek0Based ==================================================================================================
RAISERROR('Testing Nadrek0Based...',10,1) WITH NOWAIT;
DBCC FREEPROCCACHE;

--===== Start the timer
 SELECT @StartTime = GETDATE();

--===== Run the test
 SELECT csv.RowNum, split.ItemNumber, split.Item
   INTO #Nadrek0Based
   FROM dbo.CSV8K csv
  CROSS APPLY dbo.Nadrek0Based(csv.CSV,',') split;

--===== Stop the timer and record the test
 SELECT @EndTime = GETDATE();
 
 INSERT INTO dbo.TestResults
        (SplitterName, NumberOfRows, NumberOfElements, MinElementLength, MaxElementLength, Duration, MinLength, AvgLength, MaxLength)
 SELECT 'Nadrek0Based', 
        @pNumberOfRows, 
        @pNumberOfElements, 
        @pMinElementLength,
        @pMaxElementLength,
        DATEDIFF(ms,@StartTime,@EndTime)/1000.0,
        MinLength = @MinLength,
        AvgLength = @AvgLength,
        MaxLength = @MaxLength
;
--===== DelimitedSplit8K (NEW) ========================================================================================
RAISERROR('Testing DelimitedSplit8K...',10,1) WITH NOWAIT;
DBCC FREEPROCCACHE;

--===== Start the timer
 SELECT @StartTime = GETDATE();

--===== Run the test
 SELECT csv.RowNum, split.ItemNumber, split.Item
   INTO #DelimitedSplit8K
   FROM dbo.CSV8K csv
  CROSS APPLY dbo.DelimitedSplit8K(csv.CSV,',') split;

--===== Stop the timer and record the test
 SELECT @EndTime = GETDATE();
 
 INSERT INTO dbo.TestResults
        (SplitterName, NumberOfRows, NumberOfElements, MinElementLength, MaxElementLength, Duration, MinLength, AvgLength, MaxLength)
 SELECT 'DelimitedSplit8K', 
        @pNumberOfRows, 
        @pNumberOfElements, 
        @pMinElementLength,
        @pMaxElementLength,
        DATEDIFF(ms,@StartTime,@EndTime)/1000.0,
        MinLength = @MinLength,
        AvgLength = @AvgLength,
        MaxLength = @MaxLength
;
RAISERROR('==============================================================================================',10,1) WITH NOWAIT;
GO
--=====================================================================================================================
--      We're ready to rock.  Now, run all the tests automatically
--=====================================================================================================================
--===== Alert the operator as to how to check the run status
 SELECT Info = 'Please make sure you''re in the grid mode when running these tests.' UNION ALL 
 SELECT 'Click on the "Messages" tab if you want to check that the code is still running.';

--===== Declare some obviously named variables
DECLARE @SQL VARCHAR(MAX);

--===== Suppress the auto-display of rowcounts
    SET NOCOUNT ON;

--===== Create a "control" CTE and build all of the test commands from that
   WITH cteControl (NumberOfRows, NumberOfElements, MinElementLength, MaxElementLength)
     AS
        (
        --===== 1 to 10 characters per element
         SELECT 1000,    1, 1, 10 UNION ALL
         SELECT 1000,    2, 1, 10 UNION ALL
         SELECT 1000,    4, 1, 10 UNION ALL
         SELECT 1000,    8, 1, 10 UNION ALL
         SELECT 1000,   16, 1, 10 UNION ALL
         SELECT 1000,   32, 1, 10 UNION ALL
         SELECT 1000,   64, 1, 10 UNION ALL
         SELECT 1000,  128, 1, 10 UNION ALL
         SELECT 1000,  256, 1, 10 UNION ALL
         SELECT 1000,  512, 1, 10 UNION ALL
         SELECT 1000, 1150, 1, 10 UNION ALL
        --===== 10 to 20 characters per element
         SELECT 1000,    1, 10, 20 UNION ALL
         SELECT 1000,    2, 10, 20 UNION ALL
         SELECT 1000,    4, 10, 20 UNION ALL
         SELECT 1000,    8, 10, 20 UNION ALL
         SELECT 1000,   16, 10, 20 UNION ALL
         SELECT 1000,   32, 10, 20 UNION ALL
         SELECT 1000,   64, 10, 20 UNION ALL
         SELECT 1000,  128, 10, 20 UNION ALL
         SELECT 1000,  256, 10, 20 UNION ALL
         SELECT 1000,  480, 10, 20 UNION ALL
        --===== 20 to 30 characters per element
         SELECT 1000,    1, 20, 30 UNION ALL
         SELECT 1000,    2, 20, 30 UNION ALL
         SELECT 1000,    4, 20, 30 UNION ALL
         SELECT 1000,    8, 20, 30 UNION ALL
         SELECT 1000,   16, 20, 30 UNION ALL
         SELECT 1000,   32, 20, 30 UNION ALL
         SELECT 1000,   64, 20, 30 UNION ALL
         SELECT 1000,  128, 20, 30 UNION ALL
         SELECT 1000,  256, 20, 30 UNION ALL
         SELECT 1000,  290, 20, 30 UNION ALL
        --===== 30 to 40 characters per element
         SELECT 1000,    1, 30, 40 UNION ALL
         SELECT 1000,    2, 30, 40 UNION ALL
         SELECT 1000,    4, 30, 40 UNION ALL
         SELECT 1000,    8, 30, 40 UNION ALL
         SELECT 1000,   16, 30, 40 UNION ALL
         SELECT 1000,   32, 30, 40 UNION ALL
         SELECT 1000,   64, 30, 40 UNION ALL
         SELECT 1000,  128, 30, 40 UNION ALL
         SELECT 1000,  210, 30, 40 UNION ALL
        --===== 40 to 50 characters per element
         SELECT 1000,    1, 40, 50 UNION ALL
         SELECT 1000,    2, 40, 50 UNION ALL
         SELECT 1000,    4, 40, 50 UNION ALL
         SELECT 1000,    8, 40, 50 UNION ALL
         SELECT 1000,   16, 40, 50 UNION ALL
         SELECT 1000,   32, 40, 50 UNION ALL
         SELECT 1000,   64, 40, 50 UNION ALL
         SELECT 1000,  128, 40, 50 UNION ALL
         SELECT 1000,  165, 40, 50
        )
--===== Dynamically build all of the test commands from the above
 SELECT @SQL = ISNULL(@SQL,'')+
'
IF OBJECT_ID(''dbo.Csv8K'',''U'') IS NOT NULL DROP TABLE dbo.Csv8K;
 SELECT * 
   INTO dbo.Csv8K 
   FROM dbo.CreateCsv8K
        ('+CAST(NumberOfRows     AS VARCHAR(10))+', '
          +CAST(NumberOfElements AS VARCHAR(10))+', '
          +CAST(MinElementLength AS VARCHAR(10))+', '
          +CAST(MaxElementLength AS VARCHAR(10))+'); --# of Rows, # of Elements, MIN element length, MAX element length
   EXEC dbo.TestEachFunction '+CAST(NumberOfRows     AS VARCHAR(10)) +', '
                              +CAST(NumberOfElements AS VARCHAR(10))+', '
                              +CAST(MinElementLength    AS VARCHAR(10))+', '
                              +CAST(MaxElementLength    AS VARCHAR(10))+';
'
   FROM cteControl
--PRINT @SQL
--===== Run the tests
   EXEC (@SQL);

--===== Display the results in the grid so the tester can copy and paste to a spreadsheet.
 SELECT * FROM dbo.TestResults;

--===== Tell the operator what to do next
 SELECT Info = 'Test Complete.  Please copy and paste the results from the grid above to a spreadsheet' UNION ALL
 SELECT 'and email the spreadsheet to jbmoden@ameritech.net.' UNION ALL
 SELECT 'Thank you for helping me on these tests.';

 PRINT 
'Test Complete.  Please copy and paste the results from the 2nd Grid to a spreadsheet 
and email the spreadsheet to jbmoden@ameritech.net.
Thank you for helping me on these tests.'
GO
----=====================================================================================================================
----      Housekeeping (Left table TestResults standing just in case)
----=====================================================================================================================
--   DROP VIEW
--        dbo.iFunction;

--   DROP PROCEDURE
--        dbo.TestEachFunction;

--   DROP FUNCTION 
--        dbo.CreateCsv8K,
--        dbo.fnPesoSplit,
--        dbo.DelimitedSplit8K,
--        dbo.GRDelimitedSplit,
--        dbo.Nadrek0Based,
--        dbo.NadrekUnion0;

--   DROP TABLE
--        dbo.Csv8K,
--        dbo.Tally,
--        dbo.Numbers;
--GO
